#define comment ___comment
#include "core.h"
#undef comment
#include "process.h"
#include "master.h"
#include "frame.h"
#include "memory.h"
#include "symtab.h"
#include "symbol.h"
#include "srcdir.h"
#include "asm.h"
#include "bpts.h"
#include "dbmonitor/dbmonitor.h"
#include "rt.h"
#include "dbmon.h"
SRCFILE("dbmon.c")

#define	K	DB_KERNELID

void DbmonMaster::open()
{
	Menu m;

	if (pad) {
		pad->makecurrent();
		refresh();
		return;
	}
	pad = new Pad( (PadRcv*) this );
	pad->options(TRUNCATE|SORTED|ACCEPT_KBD);
	pad->name( "%s:%s", parent->name(), name());
	pad->banner( "Rtpi - %s - %s:%s", proctype(), parent->name(), name());
	m.last("ps", (Action)&DbmonMaster::refresh);
	pad->menu(m);
	pad->makecurrent();
	refresh();
}

void DbmonMaster::clean(Process *q)
{
	Process *p = 0, *c = child, *t;
	while (c) {
		if (c->isdead ||
		    (c != q && !eqstr(c->procpath, "!") && c->pad)) {
			c->userclose();
			if (!p)
				child = c->sibling;
			else
				p->sibling = c->sibling;
			t = c;
			c = c->sibling;
			delete t;
		} else {
			p = c;
			c = c->sibling;
		}
	}
}

void DbmonMaster::refresh()
{
	char s[128];
	DebugKstate dbk;
	Process *p;

	pad->clear();
	makeproc( "!", "a.out");
	::dbrequest(fd, boardid, K, DBR_GETSTATE, (int)&dbk, 0, sizeof(dbk));
	db_ntohsa(&dbk.state, 2);
	if (dbk.state != DBKS_NULL) {
		::dbrequest(fd, boardid, K, DBR_GETSNAME, (int)s, 0,sizeof(s));
		makeproc(sf("%s:-1",name()), sf("%s", s));
	}
	for( p = child; p; p = p->sibling )
		if( p->core || (p->procpath && eqstr(p->procpath,"!")))
			insert(p);
}

char *DbmonMaster::kbd(char *s)
{
	while( *s == ' ' ) ++s;
	switch( *s ){
	case '!':
		for( ++s; *s==' '; ++s ) {}
		makeproc("!", s);
		break;
	default:
		for(; *s==' '; ++s ) {}
		makeproc(sf("%s:-1",name()), s);
		break;
	}
	return 0;
}

char *DbmonMaster::help(long l)
{
	switch(l) {
		case HELP_OVERVIEW:	return "Dbmonpi Window";
		case HELP_MENU:		return "Dbmonpi Menu Bar";
		case HELP_KEY:		return "Dbmonpi Keyboard";
		case HELP_LMENU:	return "Dbmonpi Line Menus";
		default:		return 0;
	}
}

DbmonProcess::DbmonProcess(int f, int b, int pid, Process *sib, char *p,
	char *s, char *c) : Process(sib,p,s,c)
{
	fd = f;
	boardid = b;
	pipeid = pid;
}

void DbmonProcess::takeover()
{
	if( pad ){
		open();
		insert(ERRORKEY, "take over: already open");
		return;
	}
	Pick( "take over", (Action)&DbmonProcess::substitute, (long) this );
}

int DbmonProcess::accept( Action a )
{
	return a == (Action)&DbmonProcess::substitute;
}

void DbmonProcess::substitute(DbmonProcess *t)
{
	char *error;

	insert(ERRORKEY, 0);
	if( !core ){
		insert(ERRORKEY, "that ought to work - but it doesn't");
		return;
	}
	_bpts->lift();
	if( error = core->reopen(0,t->stabpath) ){
		_bpts->lay();
		insert(ERRORKEY, error);
		return;
	}
	procpath = t->procpath;
	stabpath = t->stabpath;
	comment = t->comment;
	t->isdead = 1;
	master->insert(t);
	master->insert(this);
	banner();
	if( _asm ) _asm->banner();
	if( _bpts ) _bpts->banner();
	if( memory ) memory->banner();
	if( globals ) globals->banner();
	if( srcdir ) srcdir->banner();
	core->symtab()->banner();
	pad->clear();
	_bpts->lay();
	docycle();
}

void DbmonProcess::open()
{
	Menu m, k, s;
	char *error;

	Process::openpad();
	if( core ) return;
	((DbmonMaster*)master)->clean(this);
	insert(ERRORKEY, "Checking process and symbol table...");
	core = newCore(fd, boardid, pipeid, master);
	if( error = core->open() ){
		delete core;
		core = 0;
		m.last( "open process", (Action)&DbmonProcess::open);
		pad->menu( m );
		insert(ERRORKEY, error);
		return;
	}
	insert(ERRORKEY, core->symtab()->warn());
	globals = new Globals(core);
	_asm = core->newAsm();
	_bpts = new Bpts(core);
	_bpts->lay();

	m.last( "Source",	(Action)&Process::srcfiles    );
	m.last( "Globals",	(Action)&Process::openglobals );
	m.last( "Memory",	(Action)&Process::openmemory  );
	m.last( "Assembler",	(Action)&Process::openasm     );
	m.last( "User Types",	(Action)&Process::opentypes   );
	m.last( "Journal",	(Action)&Process::openjournal );
	m.last( "Breakpoints",	(Action)&Process::openbpts    );

	s.last( "run",		(Action)&Process::go          );
	s.last( "stop",		(Action)&Process::stop        );
	s.last( "current",	(Action)&Process::currentstmt );
	s.last( "return",	(Action)&Process::pop         );
	s.last( "step into",	(Action)&Process::stepinto    );
	s.last( "step   1",	(Action)&Process::stmtstep, 1 );
	s.last( "step   5",	(Action)&Process::stmtstep, 5 );
	s.last( "step  25",	(Action)&Process::stmtstep, 25);
	s.last( "step 100",	(Action)&Process::stmtstep, 100);
	s.last( "step 500",	(Action)&Process::stmtstep, 500);
	m.last(s.index("stmt"));

	k.last( "kill?",	(Action)&DbmonProcess::destroy);
	m.last(k.index("kill"));

	pad->menu(m.index("views"));
	pad->makecurrent();
	docycle();
}

void DbmonProcess::destroy()
{
	insert(ERRORKEY, core->destroy());
	docycle();
}

Index DbmonProcess::carte()
{
	Menu m;
	if(procpath && !strcmp(procpath,"!") ){
		m.last("hang & open proc",(Action)&DbmonProcess::hangopen);
		m.last("hang & take over",(Action)&DbmonProcess::hangtakeover);
	} else {
		m.last("open process", (Action)&DbmonProcess::open);
		m.last("take over", (Action)&DbmonProcess::takeover);
	}
	return m.index();
}

void DbmonProcess::hang()
{
	char program[128];
	char *argv[10];
	
	char *from = stabpath;
	char *to = argv[0] = program;
	int i = 1;
	for(;;) {
		while(*from && *from != ' ')
			*to++ = *from++;
		*to++ = 0;
		if (*from == 0) {
			argv[i] = 0;
			break;
		} else {
			while (*++from == ' ')
				;
			if (*from)
				argv[i++] = to;
		}
	}
	char *ssave = stabpath;
	stabpath  = sf("%s", argv[0]);
	if( !dbload(fd, boardid, argv[0], argv) ) {
		insert(ERRORKEY, "can't load real-time board");
		return;
	}
	procpath = sf("%s:-1",((DbmonMaster*)master)->name());
	master->makeproc("!", ssave);
	master->insert(this);
}

void DbmonProcess::hangopen()
{
	hang();
	open();
}

void DbmonProcess::hangtakeover()
{
	hang();
	takeover();
}

const long REG_SPECIAL = 0x40000000;	// Address unlikely to be used 

Behavs DbmonCore::behavs()		{ readcontrol(); return behavetype(); }
char *DbmonCore::destroy()		{ return dbreq(DBR_KILL); }
long DbmonCore::regaddr()		{ return REG_SPECIAL; }
long DbmonCore::scratchaddr()		{ return scratch; }
char *DbmonCore::stop()			{ return dbreq(DBR_STOP); }

char *DbmonCore::run()
{
	if (pipeid && running)
		return 0;
	char *err = dbreq(DBR_RUN);
	if (pipeid && !err)
		running = 1;
	return err;
}

DbmonCore::DbmonCore(int cfd, int id, int pid)
{
	fd = cfd;
	boardid = id;
	pipeid = pid;
	dbreq(DBR_GETSCRATCH, (char*)&scratch, 0, sizeof(scratch));
	scratch = db_ntohl(scratch);
	dbreq(DBR_GETATTRIB, (char*)&attrib, 0, sizeof(attrib));
	db_ntohsa(&attrib.machine, 3);
}

char *DbmonCore::readcontrol()
{
	char *error = dbreq(DBR_GETSTATE, (char*)&state, 0, sizeof(state));
	db_ntohsa(&state.state, 2);
	if (pipeid && running && state.state != DBKS_RUNNING)
		running = 0;
	return error;
}

char *DbmonCore::dbreq(int req, char* addr,int rarg, int sz)
{
	if (pipeid && running)
		::kill(pipeid, SIGTERM);
	if (::dbrequest(fd, boardid, K, req, (int)addr, rarg, sz) == -1)
		return "dbrequest failed";
	return 0;
}

char *DbmonCore::problem()
{
	if (state.state == DBKS_NULL)
		return "no process";
	else
		return "unknown monitor state";
}

char *DbmonCore::eventname()	{ return faultname(state.code); }

char *DbmonCore::faultname(int code)
{
	if (!code)
		return "DBKS_ERROR and no code";
	 return sf("Unexpected trap or intrpt 0x%x", code);
}

char *DbmonCore::laybpt(Trap *t)
{
	if (bptsize && read(t->stmt->range.lo, t->saved, bptsize))
		return "laybpt: read failed";
	return dbreq(DBR_SETBKPT, 0, (int)t->stmt->range.lo);
}

Behavs DbmonCore::behavetype()
{
	switch( state.state ){
		default:
		case DBKS_NULL:
			return ERRORED;
		case DBKS_RUNNING:
			return ACTIVE;
		case DBKS_STOPPED:
			if (state.code == DBKC_TRAP)
				return BREAKED;
			else
				return HALTED;
		case DBKS_ERROR:
			return PENDING;
	}
}

char *DbmonCore::open()
{
	if( stabpath() ){
		stabfd = ::open(stabpath(),0);
		if( stabfd<0 ) return SysErr( "symbol tables: " );
	} else
		return "open error";
	_online = 1;
	stabfstat();
	newSymTab();
	_symtab->read();
	return readcontrol();
}

char *DbmonCore::reopen(char *, char *newstabpath)
{
	int compstabfd = -1;

	compstabfd = ::open(newstabpath, 0);
	struct stat compstabstat;
	if( compstabfd < 0 || ::fstat(compstabfd, &compstabstat) )
		return "symbol table error";
	if( compstabstat.st_mtime != stabstat.st_mtime )
		return "symbol tables differ (modified time)";
	if( compstabstat.st_size != stabstat.st_size )
		return "symbol tables differ (file size)";
	::close(compstabfd);
	return readcontrol();
}

char *DbmonCore::readwrite(long offset, char *buf, int r, int w)
{
	if (offset >= REG_SPECIAL && offset <= (REG_SPECIAL + regsize)) {
		offset -= REG_SPECIAL;
		if( r ) return dbreq(DBR_GETREGS, buf, (int)offset, r);
		return dbreq(DBR_PUTREGS, buf, (int)offset, w);
	}
	if( r ) return dbreq(DBR_READ, buf, (int)offset, r);
	return dbreq(DBR_WRITE, buf, (int)offset, w);
}

const int	STEPWAIT = 15;
char *DbmonCore::dostep(long lo, long hi, int sstep)
{
	char *error;
	long fp0, pcs, tend;
	int i;
	static int waittime[] = {1,1,2,4,6};

	if (hi) {
		tend = ::time((long*)0) + STEPWAIT;
		fp0 = fp();
		pcs = pc();
	}
	for(;;){
		if( hi && atjsr(pcs) ) {
			error = stepoverjsr();
			goto next;
		}
		if (sstep)
			error = dbreq(DBR_STEP);
		else
			error = run();
		if( !error ) {
			for (i = 0; ; i++) {
				error = readcontrol();
				if (error || behavetype() != ACTIVE)
					break;
				if (i >= 5)
					return "single step timeout";
				sleep(waittime[i]);
			}
		}
		if( !error && state.state != DBKS_STOPPED)
			error = "single step error";
next:
		if( error ) return error;
		if( !hi || (pcs = pc()) < lo || pcs >= hi
		 || ((stackdir == GROWDOWN ? (fp() > fp0) : (fp() < fp0))
		     && !atreturn(pcs)) )
			return 0;
		if( ::time((long*)0) > tend )
			return sf("single step timeout (%d secs)", STEPWAIT);
	}
}
